/**
 * @name Spin on field
 * @description Repeatedly reading a non-volatile field within the condition of an empty loop may
 *              result in an infinite loop.
 * @kind problem
 * @problem.severity warning
 * @precision medium
 * @id java/spin-on-field
 * @tags quality
 *       reliability
 *       correctness
 *       concurrency
 *       performance
 */

import java

/** A numerical comparison or an equality test. */
class ComparisonOrEqTestExpr extends Expr {
  ComparisonOrEqTestExpr() {
    this instanceof ComparisonExpr or
    this instanceof ReferenceEqualityTest
  }
}

/** An empty statement or block. */
class Empty extends Stmt {
  Empty() {
    this instanceof EmptyStmt or
    this.(BlockStmt).getNumStmt() = 0
  }
}

/** An empty loop statement. */
class EmptyLoop extends Stmt {
  EmptyLoop() {
    exists(ForStmt stmt | stmt = this |
      not exists(stmt.getAnInit()) and
      not exists(stmt.getAnUpdate()) and
      stmt.getStmt() instanceof Empty
    )
    or
    this.(WhileStmt).getStmt() instanceof Empty
    or
    this.(DoStmt).getStmt() instanceof Empty
  }

  Expr getCondition() {
    result = this.(ForStmt).getCondition() or
    result = this.(WhileStmt).getCondition() or
    result = this.(DoStmt).getCondition()
  }
}

from EmptyLoop loop, FieldAccess access, Field field, ComparisonOrEqTestExpr expr
where
  loop.getCondition() = expr and
  access.isOwnFieldAccess() and
  access.getParent() = expr and
  field = access.getVariable() and
  field.isStatic() and
  not field.isFinal() and
  not field.isVolatile() and
  field.getType() instanceof RefType
select access,
  "Spinning on " + field.getName() + " in " + loop.getEnclosingCallable().getName() + "."
